<?php

namespace App\Modules\Cliente\Controllers;

use App\Core\View;
use App\Core\Security;
use App\Modules\Cliente\Models\ClienteModel;

class ClienteController
{


    public function index()
    {

        // $carga = new ClienteModel();
        // for ($i = 0; $i < 100; $i++) {
        //     $carga->dados([
        //         "cliente_nome" => "Cliente $i",
        //         "cliente_cpf" => "00000000000 $i",
        //         "cliente_telefone" => "(63) 1234-4567 $i",
        //         "cliente_endereco" => "endereco do cliente $i",
        //     ]);
        //     $carga->cadastrar();
        // }


        $stats = (new ClienteModel())->getDashboardStats();

        (new View('Cliente/Views/index', [
            'title' => 'Dashboard de Clientes',
            'stats' => $stats
        ]))->render();
    }

    /**
     * Exibe o formulário de criação ou edição de cliente.
     * Se um ID é passado, busca os dados do cliente para preencher o formulário.
     */
    public function form($id = null)
    {
        $clientData = null;
        if ($id) {
            // Busca os dados do cliente existente para edição
            $clientData = (new ClienteModel())->pk($id)->consultar();
            if (!$clientData) {
                // Opcional: tratar cliente não encontrado, redirecionar ou mostrar erro.
                // Por enquanto, apenas exibirá um formulário vazio se não encontrar.
                // Uma melhoria seria mostrar uma mensagem de erro.
            }
        }

        (new View('Cliente/Views/form', [
            'title' => $id ? 'Editar Cliente' : 'Novo Cliente',
            'csrf_token' => Security::generateCSRF(),
            'client' => $clientData
        ]))->render();
    }

    public function list()
    {
        header('Content-Type: application/json');
        try {
            $page = $_GET['page'] ?? 1;
            $page = filter_var($page, FILTER_VALIDATE_INT, ['options' => ['default' => 1, 'min_range' => 1]]);

            $paginationData = (new ClienteModel())
                ->logMetrics(true)
                ->paginacao($page, 10); // 10 itens por página

            echo json_encode(['success' => true, 'pagination' => $paginationData]);
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    public function get($id = null)
    {
        header('Content-Type: application/json');
        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }
        try {
            $client = (new ClienteModel())->pk($id)->consultar();
            if ($client) {
                echo json_encode(['success' => true, 'data' => $client]);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Cliente não encontrado.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    public function create()
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');
        $data = json_decode(file_get_contents('php://input'), true);

        if ($this->validateData($data) !== true) {
            return;
        }

        try {
            $client = new ClienteModel();
            $mappedData = $this->mapToDbColumns($data);

            if ($client->cadastrar($mappedData)) {
                $newClient = $client->pk($client->last_id)->consultar();
                http_response_code(201);
                echo json_encode(['success' => true, 'message' => 'Cliente cadastrado com sucesso!', 'data' => $newClient ?: null]);
            } else {
                throw new \Exception("Falha ao criar cliente no banco de dados.");
            }
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    public function update($id = null)
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');

        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        $data = json_decode(file_get_contents('php://input'), true);

        if ($this->validateData($data, false) !== true) {
            return;
        }

        try {
            $client = new ClienteModel();
            $mappedData = $this->mapToDbColumns($data);
            $mappedData['cliente_id'] = $id;

            if ($client->alterar($mappedData)) {
                $updatedClient = (new ClienteModel())->pk($id)->consultar();
                echo json_encode(['success' => true, 'message' => 'Cliente atualizado com sucesso!', 'data' => $updatedClient ?: null]);
            } else {
                throw new \Exception("Nenhuma alteração foi feita. Os dados podem ser idênticos aos existentes ou o registro não foi encontrado.");
            }
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    public function delete($id = null)
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');

        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        try {
            if ((new ClienteModel())->deletar($id)) {
                echo json_encode(['success' => true, 'message' => 'Cliente excluído com sucesso!']);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Cliente não encontrado ou já excluído.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    private function mapToDbColumns($data)
    {
        $map = [
            'cliente_nome' => 'cliente_nome',
            'cliente_cpf' => 'cliente_cpf',
            'cliente_telefone' => 'cliente_telefone',
            'cliente_celular' => 'cliente_celular',
            'cliente_email' => 'cliente_email',
            'cliente_cep' => 'cliente_cep',
            'cliente_endereco' => 'cliente_endereco',
            'cliente_bairro' => 'cliente_bairro',
            'cliente_endCidade' => 'cliente_endCidade',
            'cliente_endEstado' => 'cliente_endEstado',
            'cliente_endComplemento' => 'cliente_endComplemento',
            'cliente_limiteCredito' => 'cliente_limiteCredito',
            'cliente_status' => 'cliente_status',
        ];
        $mappedData = [];
        foreach ($map as $apiKey => $dbColumn) {
            if (isset($data[$apiKey])) {
                $mappedData[$dbColumn] = $data[$apiKey];
            }
        }
        return $mappedData;
    }

    private function validateData($data, $isCreate = true)
    {
        if (empty($data['cliente_nome'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'O campo nome é obrigatório.']);
            return false;
        }

        if ($isCreate) {
            if (empty($data['cliente_cpf'])) {
                http_response_code(400);
                echo json_encode(['success' => false, 'message' => 'O campo CPF/CNPJ é obrigatório.']);
                return false;
            }

            $cpfCnpj = preg_replace('/[^0-9]/', '', $data['cliente_cpf']);

            if (strlen($cpfCnpj) === 11) {
                // CPF validation logic (you can add a more robust validation)
                if (!preg_match('/^\d{11}$/', $cpfCnpj)) {
                    http_response_code(400);
                    echo json_encode(['success' => false, 'message' => 'Formato de CPF inválido.']);
                    return false;
                }
            } elseif (strlen($cpfCnpj) === 14) {
                // CNPJ validation logic (you can add a more robust validation)
                if (!preg_match('/^\d{14}$/', $cpfCnpj)) {
                    http_response_code(400);
                    echo json_encode(['success' => false, 'message' => 'Formato de CNPJ inválido.']);
                    return false;
                }
            } else {
                http_response_code(400);
                echo json_encode(['success' => false, 'message' => 'CPF/CNPJ deve ter 11 ou 14 dígitos.']);
                return false;
            }

            if (ClienteModel::query()->and('cliente_cpf', $data['cliente_cpf'])->consultar()) {
                http_response_code(409); // Conflict
                echo json_encode(['success' => false, 'message' => 'CPF/CNPJ já cadastrado.']);
                return false;
            }
        }
        return true;
    }

}
