<?php

namespace App\Modules\ContasBancarias\Controllers;

use App\Core\View;
use App\Core\Security;
use App\Modules\ContasBancarias\Models\ContasBancariasModel;

class ContasBancariasController
{


    public function index()
    {
        $stats = (new ContasBancariasModel())->getDashboardStats();

        (new View('ContasBancarias/Views/index', [
            'title' => 'Contas Bancárias',
            'stats' => $stats
        ]))->render();
    }

    /**
     * Exibe o formulário de criação ou edição de cliente.
     * Se um ID é passado, busca os dados do cliente para preencher o formulário.
     */
    public function form($id = null)
    {
        $accountData = null;
        if ($id) {
            // Busca os dados do cliente existente para edição
            $accountData = (new ContasBancariasModel())->pk($id)->consultar();
            if (!$accountData) {
                // Opcional: tratar cliente não encontrado, redirecionar ou mostrar erro.
                // Por enquanto, apenas exibirá um formulário vazio se não encontrar.
                // Uma melhoria seria mostrar uma mensagem de erro.
            }
        }

        (new View('ContasBancarias/Views/form', [
            'title' => $id ? 'Editar Conta Bancária' : 'Nova Conta',
            'csrf_token' => Security::generateCSRF(),
            'conta' => $accountData
        ]))->render();
    }

    public function list()
    {

        header('Content-Type: application/json');
        try {
            $page = $_GET['page'] ?? 1;
            $page = filter_var($page, FILTER_VALIDATE_INT, ['options' => ['default' => 1, 'min_range' => 1]]);

            $paginationData = (new ContasBancariasModel())
                ->logMetrics(true)
                ->paginacao($page, 10); // 10 itens por página

            echo json_encode(['success' => true, 'pagination' => $paginationData]);
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    public function get($id = null)
    {
        header('Content-Type: application/json');
        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }
        try {
            $customer = (new ContasBancariasModel())->pk($id)->consultar();
            if ($customer) {
                echo json_encode(['success' => true, 'data' => $customer]);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Conta Bancária não encontrada.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    public function create()
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');
        $data = json_decode(file_get_contents('php://input'), true);

        if ($this->validateData($data) !== true) {
            return;
        }

        try {
            $customer = new ContasBancariasModel();
            $mappedData = $this->mapToDbColumns($data);

            if ($customer->cadastrar($mappedData)) {
                $newContasBancarias = $customer->pk($customer->last_id)->consultar();
                http_response_code(201);
                echo json_encode(['success' => true, 'message' => 'Conta Bancária cadastrada com sucesso!', 'data' => $newContasBancarias ?: null]);
            } else {
                throw new \Exception("Falha ao criar conta bancária no banco de dados.");
            }
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    public function update($id = null)
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');

        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        $data = json_decode(file_get_contents('php://input'), true);

        if ($this->validateData($data, false) !== true) {
            return;
        }

        try {
            $customer = new ContasBancariasModel();
            $mappedData = $this->mapToDbColumns($data);
            $mappedData['conta_bancaria_id'] = $id;

            if ($customer->alterar($mappedData)) {
                $updatedContasBancarias = (new ContasBancariasModel())->pk($id)->consultar();
                echo json_encode(['success' => true, 'message' => 'Dados da Conta Bancária atualizada com sucesso!', 'data' => $updatedContasBancarias ?: null]);
            } else {
                throw new \Exception("Nenhuma alteração foi feita. Os dados podem ser idênticos aos existentes ou o registro não foi encontrado.");
            }
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    public function delete($id = null)
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');

        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        try {
            if ((new ContasBancariasModel())->deletar($id)) {
                echo json_encode(['success' => true, 'message' => 'Conta Bancária excluída com sucesso!']);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Conta Bancária não encontrada ou já excluído.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    private function mapToDbColumns($data)
    {
        $map = [
            'name' => 'conta_bancaria_nome',
            'account' => 'conta_bancaria_conta',
            'type' => 'conta_bancaria_tipo',
            'holder' => 'conta_bancaria_titular',
            'status' => 'conta_bancaria_status',
            'observation' => 'conta_bancaria_observacao'
        ];
        $mappedData = [];
        foreach ($map as $apiKey => $dbColumn) {
            if (isset($data[$apiKey])) {
                $mappedData[$dbColumn] = $data[$apiKey];
            }
        }
        return $mappedData;
    }

    private function validateData($data, $isCreate = true)
    {
        if (empty($data['name'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'O campo nome é obrigatório.']);
            return false;
        }

        return true;
    }

}
