<?php

namespace App\Modules\Estoque\CategoriaProduto\Controllers;

use App\Core\View;
use App\Core\Security;
use App\Modules\Estoque\CategoriaProduto\Models\CategoriaProdutoModel;

class CategoriaProdutoController
{

    public function index()
    {
        $stats = (new CategoriaProdutoModel())->getDashboardStats();

        (new View('Estoque/CategoriaProduto/Views/index', [
            'title' => 'Categorias de Produtos',
            'stats' => $stats
        ]))->render();
    }

    public function form($id = null)
    {
        $data = null;
        if ($id) {
            $data = (new CategoriaProdutoModel())->pk($id)->consultar();
        }

        (new View('Estoque/CategoriaProduto/Views/form', [
            'title' => $id ? 'Editar Categoria' : 'Nova Categoria',
            'csrf_token' => Security::generateCSRF(),
            'categoria' => $data
        ]))->render();
    }

    public function list()
    {
        header('Content-Type: application/json');
        try {
            $page = $_GET['page'] ?? 1;
            $page = filter_var($page, FILTER_VALIDATE_INT, ['options' => ['default' => 1, 'min_range' => 1]]);

            $limit = $_GET['limit'] ?? 10;
            $limit = filter_var($limit, FILTER_VALIDATE_INT, ['options' => ['default' => 10, 'min_range' => 1]]);

            $paginationData = (new CategoriaProdutoModel())
                ->logMetrics(true)
                ->paginacao($page, $limit);

            echo json_encode(['success' => true, 'pagination' => $paginationData]);
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    public function get($id = null)
    {
        header('Content-Type: application/json');
        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }
        try {
            $data = (new CategoriaProdutoModel())->pk($id)->consultar();
            if ($data) {
                echo json_encode(['success' => true, 'data' => $data]);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Categoria não encontrada.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    public function create()
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');
        $data = json_decode(file_get_contents('php://input'), true);

        if (empty($data['produto_categoria_nome'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'O nome da categoria é obrigatório.']);
            return;
        }

        try {
            $model = new CategoriaProdutoModel();
            $mappedData = [
                'produto_categoria_nome' => $data['produto_categoria_nome']
            ];

            if ($model->cadastrar($mappedData)) {
                $newCategory = $model->pk($model->last_id)->consultar();
                http_response_code(201);
                echo json_encode(['success' => true, 'message' => 'Categoria criada com sucesso!', 'data' => $newCategory]);
            } else {
                throw new \Exception("Falha ao criar categoria.");
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    public function update($id = null)
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');

        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        $data = json_decode(file_get_contents('php://input'), true);

        if (empty($data['produto_categoria_nome'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'O nome da categoria é obrigatório.']);
            return;
        }

        try {
            $model = new CategoriaProdutoModel();
            $mappedData = [
                'produto_categoria_id' => $id,
                'produto_categoria_nome' => $data['produto_categoria_nome']
            ];

            if ($model->alterar($mappedData)) {
                $updatedCategory = (new CategoriaProdutoModel())->pk($id)->consultar();
                echo json_encode(['success' => true, 'message' => 'Categoria atualizada com sucesso!', 'data' => $updatedCategory]);
            } else {
                throw new \Exception("Nenhuma alteração realizada.");
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    public function delete($id = null)
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');

        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        try {
            if ((new CategoriaProdutoModel())->deletar($id)) {
                echo json_encode(['success' => true, 'message' => 'Categoria excluída com sucesso!']);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Categoria não encontrada.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }
}
