<?php

namespace App\Modules\Estoque\UnidadeMedida\Controllers;

use App\Core\View;
use App\Modules\Estoque\UnidadeMedida\Models\UnidadeMedidaModel;

class UnidadeMedidaController
{

    public function index()
    {
        $stats = (new UnidadeMedidaModel())->getDashboardStats();

        (new View('Estoque/UnidadeMedida/Views/index', [
            'title' => 'Unidades de Medida',
            'stats' => $stats
        ]))->render();
    }

    public function form($id = null)
    {
        $data = null;
        if ($id) {
            $data = (new UnidadeMedidaModel())->pk($id)->consultar();
        }

        (new View('Estoque/UnidadeMedida/Views/form', [
            'title' => $id ? 'Editar Unidade' : 'Nova Unidade',
            'unidade' => $data
        ]))->render();
    }

    public function list()
    {
        header('Content-Type: application/json');
        try {
            $page = $_GET['page'] ?? 1;
            $page = filter_var($page, FILTER_VALIDATE_INT, ['options' => ['default' => 1, 'min_range' => 1]]);

            $limit = $_GET['limit'] ?? 10;
            $limit = filter_var($limit, FILTER_VALIDATE_INT, ['options' => ['default' => 10, 'min_range' => 1]]);

            $paginationData = (new UnidadeMedidaModel())
                ->logMetrics(true)
                ->paginacao($page, $limit);

            echo json_encode(['success' => true, 'pagination' => $paginationData]);
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    public function get($id = null)
    {
        header('Content-Type: application/json');
        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }
        try {
            $data = (new UnidadeMedidaModel())->pk($id)->consultar();
            if ($data) {
                echo json_encode(['success' => true, 'data' => $data]);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Unidade não encontrada.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    public function create()
    {
        header('Content-Type: application/json');
        $data = json_decode(file_get_contents('php://input'), true);

        if (empty($data['unidade_medida_nome'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'O nome da unidade é obrigatório.']);
            return;
        }

        try {
            $model = new UnidadeMedidaModel();
            $mappedData = [
                'unidade_medida_nome' => $data['unidade_medida_nome']
            ];

            if ($model->cadastrar($mappedData)) {
                $newItem = $model->pk($model->last_id)->consultar();
                http_response_code(201);
                echo json_encode(['success' => true, 'message' => 'Unidade criada com sucesso!', 'data' => $newItem]);
            } else {
                throw new \Exception("Falha ao criar unidade.");
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    public function update($id = null)
    {
        header('Content-Type: application/json');

        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        $data = json_decode(file_get_contents('php://input'), true);

        if (empty($data['unidade_medida_nome'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'O nome da unidade é obrigatório.']);
            return;
        }

        try {
            $model = new UnidadeMedidaModel();
            $mappedData = [
                'unidade_medida_id' => $id,
                'unidade_medida_nome' => $data['unidade_medida_nome']
            ];

            if ($model->alterar($mappedData)) {
                $updatedItem = (new UnidadeMedidaModel())->pk($id)->consultar();
                echo json_encode(['success' => true, 'message' => 'Unidade atualizada com sucesso!', 'data' => $updatedItem]);
            } else {
                throw new \Exception("Nenhuma alteração realizada.");
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    public function delete($id = null)
    {
        header('Content-Type: application/json');

        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        try {
            if ((new UnidadeMedidaModel())->deletar($id)) {
                echo json_encode(['success' => true, 'message' => 'Unidade excluída com sucesso!']);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Unidade não encontrada.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }
}
