<?php

namespace App\Modules\Estoque\Produtos\Controllers;

use App\Core\View;
use App\Modules\Estoque\Produtos\Models\ProdutosModel;

class ProdutosController
{
    public function index()
    {
        $stats = (new ProdutosModel())->getDashboardStats();

        (new View('Estoque/produtos/Views/index', [
            'title' => 'Gerenciar Produtos',
            'stats' => $stats
        ]))->render();
    }

    public function form($id = null)
    {
        $data = null;
        if ($id) {
            $data = (new ProdutosModel())->pk($id)->consultar();
        }

        // Dependencies are now fetched via JS

        (new View('Estoque/produtos/Views/form', [
            'title' => $id ? 'Editar Produto' : 'Novo Produto',
            'produto' => $data
        ]))->render();
    }

    public function list()
    {
        header('Content-Type: application/json');
        try {
            $page = $_GET['page'] ?? 1;
            $page = filter_var($page, FILTER_VALIDATE_INT, ['options' => ['default' => 1, 'min_range' => 1]]);

            $model = new ProdutosModel();
            
            // Limit increased to 10 for consistency
            $paginationData = $model->paginacao($page, 10);

            echo json_encode(['success' => true, 'pagination' => $paginationData]);
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    public function get($id = null)
    {
        header('Content-Type: application/json');
        if (!$id) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        try {
            $data = (new ProdutosModel())->pk($id)->consultar();
            if ($data) {
                echo json_encode(['success' => true, 'data' => $data]);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Produto não encontrado.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    public function create()
    {
        header('Content-Type: application/json');
        $data = json_decode(file_get_contents('php://input'), true);

        if (empty($data['produto_descricao'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Descrição é obrigatória.']);
            return;
        }

        try {
            $model = new ProdutosModel();
            
            // Debug: Check if colunas is accessible
            error_log("Colunas property: " . print_r($model->colunas, true));
            
            // Define allowed columns directly to avoid null issues
            $allowed = [
                'produto_descricao',
                'produto_quantidade',
                'produto_valor',
                'produto_categoria',
                'produto_infoAdcional',
                'produto_aplicacao',
                'produto_estoqueAtual',
                'produto_ncm',
                'produto_valorVenda',
                'produto_csosn',
                'produto_margemLucro',
                'produto_unidade_medida'
            ];
            
            $insertData = [];
            foreach ($allowed as $col) {
                if (isset($data[$col])) {
                    // Skip empty foreign keys to avoid constraint violations
                    if (($col === 'produto_categoria' || $col === 'produto_unidade_medida') && empty($data[$col])) {
                        continue;
                    }
                    $insertData[$col] = $data[$col];
                }
            }

            if ($model->cadastrar($insertData)) {
                $newItem = $model->pk($model->last_id)->consultar();
                http_response_code(201);
                echo json_encode(['success' => true, 'message' => 'Produto criado com sucesso!', 'data' => $newItem]);
            } else {
                throw new \Exception("Erro ao criar produto.");
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    public function update($id = null)
    {
        header('Content-Type: application/json');

        if (!$id) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        $data = json_decode(file_get_contents('php://input'), true);

        try {
            $model = new ProdutosModel();
            
            // Define allowed columns directly to avoid null issues
            $allowed = [
                'produto_descricao',
                'produto_quantidade',
                'produto_valor',
                'produto_categoria',
                'produto_infoAdcional',
                'produto_aplicacao',
                'produto_estoqueAtual',
                'produto_ncm',
                'produto_valorVenda',
                'produto_csosn',
                'produto_margemLucro',
                'produto_unidade_medida'
            ];
            
            $updateData = ['produto_id' => $id];
            
            foreach ($allowed as $col) {
                if (isset($data[$col])) {
                    // Skip empty foreign keys to avoid constraint violations
                    if (($col === 'produto_categoria' || $col === 'produto_unidade_medida') && empty($data[$col])) {
                        continue;
                    }
                    $updateData[$col] = $data[$col];
                }
            }

            if ($model->alterar($updateData)) {
                $updatedItem = $model->pk($id)->consultar();
                echo json_encode(['success' => true, 'message' => 'Produto atualizado com sucesso!', 'data' => $updatedItem]);
            } else {
                 echo json_encode(['success' => true, 'message' => 'Produto atualizado com sucesso!', 'data' => $data]);
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    public function delete($id = null)
    {
        header('Content-Type: application/json');

        if (!$id) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        try {
            if ((new ProdutosModel())->deletar($id)) {
                echo json_encode(['success' => true, 'message' => 'Produto excluído com sucesso!']);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Produto não encontrado.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => $e->getMessage()]);
        }
    }
}
