<?php

namespace App\Modules\Financeiro\ContasPagar\Controllers;

use App\Core\View;
use App\Core\Security;
use App\Modules\Financeiro\ContasPagar\Models\ContaPagarModel;
use App\Modules\Financeiro\PlanoDeContas\Models\PlanoDeContasModel;
use App\Modules\Fornecedor\Models\FornecedorModel;

class ContasPagarController
{
    public function index()
    {
        (new View('Financeiro/ContasPagar/Views/index', [
            'title' => 'Contas a Pagar'
        ]))->render();
    }

    public function form($id = null)
    {
        $data = null;
        if ($id) {
            $data = (new ContaPagarModel())->pk($id)->consultar();
        }

        (new View('Financeiro/ContasPagar/Views/form', [
            'title' => $id ? 'Editar Conta a Pagar' : 'Nova Conta a Pagar',
            'csrf_token' => Security::generateCSRF(),
            'data' => $data,
            'planos_de_contas' => (new PlanoDeContasModel())->and('plano_conta_tipo_conta', 'despesa')->consultar(),
            'fornecedores' => (new FornecedorModel())->consultar()
        ]))->render();
    }

    public function list()
    {
        header('Content-Type: application/json');
        try {
            $page = $_GET['page'] ?? 1;
            $page = filter_var($page, FILTER_VALIDATE_INT, ['options' => ['default' => 1, 'min_range' => 1]]);
            
            $paginationData = (new ContaPagarModel())
                ->logMetrics(true)
                ->paginacao($page, 10);

            echo json_encode(['success' => true, 'pagination' => $paginationData]);
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get($id = null)
    {
        header('Content-Type: application/json');
        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }
        try {
            $data = (new ContaPagarModel())->pk($id)->consultar();
            if ($data) {
                echo json_encode(['success' => true, 'data' => $data]);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Conta a pagar não encontrada.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function create()
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');
        $data = json_decode(file_get_contents('php://input'), true);

        if ($this->validateData($data) !== true) {
            return;
        }

        try {
            $model = new ContaPagarModel();
            $mappedData = $this->mapToDbColumns($data);

            if ($model->cadastrar($mappedData)) {
                $newData = $model->pk($model->last_id)->consultar();
                http_response_code(201);
                echo json_encode(['success' => true, 'message' => 'Conta a pagar cadastrada com sucesso!', 'data' => $newData ?: null]);
            } else {
                throw new \Exception("Falha ao criar conta a pagar no banco de dados.");
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function update($id = null)
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');

        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        $data = json_decode(file_get_contents('php://input'), true);

        if ($this->validateData($data, false) !== true) {
            return;
        }

        try {
            $model = new ContaPagarModel();
            $mappedData = $this->mapToDbColumns($data);
            $mappedData['conta_pagar_id'] = $id;

            if ($model->alterar($mappedData)) {
                $updatedData = (new ContaPagarModel())->pk($id)->consultar();
                echo json_encode(['success' => true, 'message' => 'Conta a pagar atualizada com sucesso!', 'data' => $updatedData ?: null]);
            } else {
                throw new \Exception("Nenhuma alteração foi feita.");
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function delete($id = null)
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');

        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        try {
            if ((new ContaPagarModel())->deletar($id)) {
                echo json_encode(['success' => true, 'message' => 'Conta a pagar excluída com sucesso!']);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Conta a pagar não encontrada ou já excluída.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    private function mapToDbColumns($data)
    {
        $map = [
            'conta_pagar_valor' => 'conta_pagar_valor',
            'conta_pagar_vencimento' => 'conta_pagar_vencimento',
            'conta_pagar_plano_contas' => 'conta_pagar_plano_contas',
            'conta_pagar_descricao' => 'conta_pagar_descricao',
            'conta_pagar_status' => 'conta_pagar_status',
            'conta_pagar_data_pagamento' => 'conta_pagar_data_pagamento',
            'conta_pagar_fornecedor' => 'conta_pagar_fornecedor'
        ];
        $mappedData = [];
        foreach ($map as $apiKey => $dbColumn) {
            if (isset($data[$apiKey])) {
                $mappedData[$dbColumn] = $data[$apiKey] ?: null;
            }
        }
        return $mappedData;
    }

    private function validateData($data, $isCreate = true)
    {
        if (empty($data['conta_pagar_valor'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'O campo valor é obrigatório.']);
            return false;
        }

        if (empty($data['conta_pagar_vencimento'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'O campo vencimento é obrigatório.']);
            return false;
        }
        
        if (empty($data['conta_pagar_plano_contas'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'O campo plano de contas é obrigatório.']);
            return false;
        }

        return true;
    }
}
