<?php

namespace App\Modules\Financeiro\ContasReceber\Controllers;

use App\Core\View;
use App\Core\Security;
use App\Modules\Financeiro\ContasReceber\Models\ContaReceberModel;
use App\Modules\Financeiro\PlanoDeContas\Models\PlanoDeContasModel;
use App\Modules\Cliente\Models\ClienteModel;

class ContasReceberController
{
    public function index()
    {
        (new View('Financeiro/ContasReceber/Views/index', [
            'title' => 'Contas a Receber'
        ]))->render();
    }

    public function form($id = null)
    {
        $data = null;
        if ($id) {
            $data = (new ContaReceberModel())->pk($id)->consultar();
        }

        (new View('Financeiro/ContasReceber/Views/form', [
            'title' => $id ? 'Editar Conta a Receber' : 'Nova Conta a Receber',
            'csrf_token' => Security::generateCSRF(),
            'data' => $data,
            'planos_de_contas' => (new PlanoDeContasModel())->and('plano_conta_tipo_conta', 'receita')->consultar(),
            'clientes' => (new ClienteModel())->consultar()
        ]))->render();
    }

    public function list()
    {
        header('Content-Type: application/json');
        try {
            $page = $_GET['page'] ?? 1;
            $page = filter_var($page, FILTER_VALIDATE_INT, ['options' => ['default' => 1, 'min_range' => 1]]);
            
            $paginationData = (new ContaReceberModel())
                ->logMetrics(true)
                ->paginacao($page, 10);

            echo json_encode(['success' => true, 'pagination' => $paginationData]);
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function get($id = null)
    {
        header('Content-Type: application/json');
        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }
        try {
            $data = (new ContaReceberModel())->pk($id)->consultar();
            if ($data) {
                echo json_encode(['success' => true, 'data' => $data]);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Conta a receber não encontrada.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function create()
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');
        $data = json_decode(file_get_contents('php://input'), true);

        if ($this->validateData($data) !== true) {
            return;
        }

        try {
            $model = new ContaReceberModel();
            $mappedData = $this->mapToDbColumns($data);

            if ($model->cadastrar($mappedData)) {
                $newData = $model->pk($model->last_id)->consultar();
                http_response_code(201);
                echo json_encode(['success' => true, 'message' => 'Conta a receber cadastrada com sucesso!', 'data' => $newData ?: null]);
            } else {
                throw new \Exception("Falha ao criar conta a receber no banco de dados.");
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function update($id = null)
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');

        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        $data = json_decode(file_get_contents('php://input'), true);

        if ($this->validateData($data, false) !== true) {
            return;
        }

        try {
            $model = new ContaReceberModel();
            $mappedData = $this->mapToDbColumns($data);
            $mappedData['conta_receber_id'] = $id;

            if ($model->alterar($mappedData)) {
                $updatedData = (new ContaReceberModel())->pk($id)->consultar();
                echo json_encode(['success' => true, 'message' => 'Conta a receber atualizada com sucesso!', 'data' => $updatedData ?: null]);
            } else {
                throw new \Exception("Nenhuma alteração foi feita.");
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    public function delete($id = null)
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');

        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        try {
            if ((new ContaReceberModel())->deletar($id)) {
                echo json_encode(['success' => true, 'message' => 'Conta a receber excluída com sucesso!']);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Conta a receber não encontrada ou já excluída.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);
            echo json_encode([
                'success' => false,
                'message' => $e->getMessage()
            ]);
        }
    }

    private function mapToDbColumns($data)
    {
        $map = [
            'conta_receber_valor' => 'conta_receber_valor',
            'conta_receber_vencimento' => 'conta_receber_vencimento',
            'conta_receber_plano_contas' => 'conta_receber_plano_contas',
            'conta_receber_descricao' => 'conta_receber_descricao',
            'conta_receber_status' => 'conta_receber_status',
            'conta_receber_data_pagamento' => 'conta_receber_data_pagamento',
            'conta_receber_cliente' => 'conta_receber_cliente'
        ];
        $mappedData = [];
        foreach ($map as $apiKey => $dbColumn) {
            if (isset($data[$apiKey])) {
                $mappedData[$dbColumn] = $data[$apiKey] ?: null;
            }
        }
        return $mappedData;
    }

    private function validateData($data, $isCreate = true)
    {
        if (empty($data['conta_receber_valor'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'O campo valor é obrigatório.']);
            return false;
        }

        if (empty($data['conta_receber_vencimento'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'O campo vencimento é obrigatório.']);
            return false;
        }
        
        if (empty($data['conta_receber_plano_contas'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'O campo plano de contas é obrigatório.']);
            return false;
        }
        
        if (empty($data['conta_receber_cliente'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'O campo cliente é obrigatório.']);
            return false;
        }

        return true;
    }
}
