<?php

namespace App\Modules\Financeiro\Planodecontas\Controllers;

use App\Core\View;
use App\Core\Security;
use App\Modules\Financeiro\Planodecontas\Models\PlanodecontasModel;

class PlanodecontasController
{


    public function index()
    {
        $stats = (new PlanodecontasModel())->getDashboardStats();

        (new View('Financeiro/Planodecontas/Views/index', [
            'title' => 'Plano de Contas',
            'stats' => $stats
        ]))->render();
    }

    /**
     * Exibe o formulário de criação ou edição de cliente.
     * Se um ID é passado, busca os dados do cliente para preencher o formulário.
     */
    public function form($id = null)
    {
        $planoDeContasData = null;
        if ($id) {
            // Busca os dados do cliente existente para edição
            $planoDeContasData = (new PlanodecontasModel())->pk($id)->consultar();
            if (!$planoDeContasData) {
                // Opcional: tratar cliente não encontrado, redirecionar ou mostrar erro.
                // Por enquanto, apenas exibirá um formulário vazio se não encontrar.
                // Uma melhoria seria mostrar uma mensagem de erro.
            }
        }

        (new View('Financeiro/Planodecontas/Views/form', [
            'title' => $id ? 'Editar Plano de Conta' : 'Novo Plano de Conta',
            'csrf_token' => Security::generateCSRF(),
            'plano_de_contas' => $planoDeContasData
        ]))->render();
    }

    public function list()
    {
        header('Content-Type: application/json');
        try {
            $page = $_GET['page'] ?? 1;
            $page = filter_var($page, FILTER_VALIDATE_INT, ['options' => ['default' => 1, 'min_range' => 1]]);


            $paginationData = (new PlanodecontasModel())
                ->paginacao($page, 10); // 10 itens por página            

            echo json_encode(['success' => true, 'pagination' => $paginationData]);
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    public function get($id = null)
    {
        header('Content-Type: application/json');
        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }
        try {
            $planoDeContas = (new PlanodecontasModel())->pk($id)->consultar();
            if ($planoDeContas) {
                echo json_encode(['success' => true, 'data' => $planoDeContas]);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Plano de Conta não encontrado.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    public function create()
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');
        $data = json_decode(file_get_contents('php://input'), true);

        if ($this->validateData($data) !== true) {
            return;
        }

        try {
            $planoDeContas = new PlanodecontasModel();
            $mappedData = $this->mapToDbColumns($data);

            if ($planoDeContas->cadastrar($mappedData)) {
                $newPlanoDeContas = $planoDeContas->pk($planoDeContas->last_id)->consultar();
                http_response_code(201);
                echo json_encode(['success' => true, 'message' => 'Plano de Conta cadastrado com sucesso!', 'data' => $newPlanoDeContas ?: null]);
            } else {
                throw new \Exception("Falha ao criar plano de conta no banco de dados.");
            }
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    public function update($id = null)
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');

        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        $data = json_decode(file_get_contents('php://input'), true);

        if ($this->validateData($data, false) !== true) {
            return;
        }

        try {
            $planoDeContas = new PlanodecontasModel();
            $mappedData = $this->mapToDbColumns($data);
            $mappedData['plano_conta_id'] = $id;

            if ($planoDeContas->alterar($mappedData)) {
                $updatedPlanoDeContas = (new PlanodecontasModel())->pk($id)->consultar();
                echo json_encode(['success' => true, 'message' => 'Plano de Conta atualizado com sucesso!', 'data' => $updatedPlanoDeContas ?: null]);
            } else {
                throw new \Exception("Nenhuma alteração foi feita. Os dados podem ser idênticos aos existentes ou o registro não foi encontrado.");
            }
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    public function delete($id = null)
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');

        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        try {
            if ((new PlanodecontasModel())->deletar($id)) {
                echo json_encode(['success' => true, 'message' => 'Plano de Conta excluído com sucesso!']);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Plano de Conta não encontrado ou já excluído.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }


    private function mapToDbColumns($data)
    {
        $map = [
            'name' => 'plano_conta_nome',
            'status' => 'plano_conta_status'
        ];
        $mappedData = [];
        foreach ($map as $apiKey => $dbColumn) {
            if (isset($data[$apiKey])) {
                $mappedData[$dbColumn] = $data[$apiKey];
            }
        }
        return $mappedData;
    }

    private function validateData($data, $isCreate = true)
    {
        if (empty($data['name'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'O campo nome é obrigatório.']);
            return false;
        }

        if (empty($data['status'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'O campo status é obrigatório.']);
            return false;
        }

        return true;
    }

}