<?php

namespace App\Modules\Fornecedor\Controllers;

use App\Core\View;
use App\Core\Security;
use App\Modules\Fornecedor\Models\FornecedorModel;

class FornecedorController
{


    public function index()
    {
        $stats = (new FornecedorModel())->getDashboardStats();

        (new View('Fornecedor/Views/index', [
            'title' => 'Dashboard de Fornecedores',
            'stats' => $stats
        ]))->render();
    }

    /**
     * Exibe o formulário de criação ou edição de cliente.
     * Se um ID é passado, busca os dados do cliente para preencher o formulário.
     */
    public function form($id = null)
    {
        $fornecedorData = null;
        if ($id) {
            // Busca os dados do cliente existente para edição
            $fornecedorData = (new FornecedorModel())->pk($id)->consultar();
            if (!$fornecedorData) {
                // Opcional: tratar cliente não encontrado, redirecionar ou mostrar erro.
                // Por enquanto, apenas exibirá um formulário vazio se não encontrar.
                // Uma melhoria seria mostrar uma mensagem de erro.
            }
        }

        (new View('Fornecedor/Views/form', [
            'title' => $id ? 'Editar Fornecedor' : 'Novo Fornecedor',
            'csrf_token' => Security::generateCSRF(),
            'fornecedor' => $fornecedorData
        ]))->render();
    }

    public function list()
    {
        header('Content-Type: application/json');
        try {
            $page = $_GET['page'] ?? 1;
            $page = filter_var($page, FILTER_VALIDATE_INT, ['options' => ['default' => 1, 'min_range' => 1]]);


            $paginationData = (new FornecedorModel())
                ->paginacao($page, 10); // 10 itens por página            

            echo json_encode(['success' => true, 'pagination' => $paginationData]);
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    public function get($id = null)
    {
        header('Content-Type: application/json');
        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }
        try {
            $fornecedor = (new FornecedorModel())->pk($id)->consultar();
            if ($fornecedor) {
                echo json_encode(['success' => true, 'data' => $fornecedor]);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Fornecedor não encontrado.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    public function create()
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');
        $data = json_decode(file_get_contents('php://input'), true);

        if ($this->validateData($data) !== true) {
            return;
        }

        try {
            $fornecedor = new FornecedorModel();
            $mappedData = $this->mapToDbColumns($data);

            if ($fornecedor->cadastrar($mappedData)) {
                $newFornecedor = $fornecedor->pk($fornecedor->last_id)->consultar();
                http_response_code(201);
                echo json_encode(['success' => true, 'message' => 'Fornecedor cadastrado com sucesso!', 'data' => $newFornecedor ?: null]);
            } else {
                throw new \Exception("Falha ao criar cliente no banco de dados.");
            }
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    public function update($id = null)
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');

        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        $data = json_decode(file_get_contents('php://input'), true);

        if ($this->validateData($data, false) !== true) {
            return;
        }

        try {
            $fornecedor = new FornecedorModel();
            $mappedData = $this->mapToDbColumns($data);
            $mappedData['fornecedor_id'] = $id;

            if ($fornecedor->alterar($mappedData)) {
                $updatedFornecedor = (new FornecedorModel())->pk($id)->consultar();
                echo json_encode(['success' => true, 'message' => 'Fornecedor atualizado com sucesso!', 'data' => $updatedFornecedor ?: null]);
            } else {
                throw new \Exception("Nenhuma alteração foi feita. Os dados podem ser idênticos aos existentes ou o registro não foi encontrado.");
            }
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }

    public function delete($id = null)
    {
        Security::checkCSRFJson();
        header('Content-Type: application/json');

        if (!is_numeric($id)) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'ID inválido.']);
            return;
        }

        try {
            if ((new FornecedorModel())->deletar($id)) {
                echo json_encode(['success' => true, 'message' => 'Fornecedor excluído com sucesso!']);
            } else {
                http_response_code(404);
                echo json_encode(['success' => false, 'message' => 'Fornecedor não encontrado ou já excluído.']);
            }
        } catch (\Exception $e) {
            http_response_code(500);

            $messages = explode('||', $e->getMessage(), 2);
            $userMessage = $messages[0];
            $devMessage = $messages[1] ?? 'No developer message available.';

            echo json_encode([
                'success' => false,
                'message' => $userMessage,
                'messageDev' => $devMessage
            ]);
        }
    }


    private function mapToDbColumns($data)
    {
        $map = [
            'name' => 'fornecedor_nome',
            'cnpj' => 'fornecedor_cnpj',
            'email' => 'fornecedor_email',
            'phone' => 'fornecedor_telefone',
            'status' => 'fornecedor_status'
        ];
        $mappedData = [];
        foreach ($map as $apiKey => $dbColumn) {
            if (isset($data[$apiKey])) {
                $mappedData[$dbColumn] = $data[$apiKey];
            }
        }
        return $mappedData;
    }

    private function validateData($data, $isCreate = true)
    {
        if (empty($data['name'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'O campo nome é obrigatório.']);
            return false;
        }

        if ($isCreate) {
            if (empty($data['cnpj'])) {
                http_response_code(400);
                echo json_encode(['success' => false, 'message' => 'O campo CNPJ é obrigatório.']);
                return false;
            }

            if (FornecedorModel::query()->and('fornecedor_cnpj', $data['cnpj'])->consultar()) {
                http_response_code(409); // Conflict
                echo json_encode(['success' => false, 'message' => 'CNPJ já cadastrado.']);
                return false;
            }
        }
        return true;
    }

}