# 📖 Guia de Uso: ORM System (no vscode use o comando control+shift+v)

Esta documentação detalha como utilizar o motor de banco de dados e a classe de upload integrados.

## 📂 Estrutura de Pastas
* **Modules/Models**: Definição das tabelas (Ex: MinhaClasse.php).
* **Modules/Controllers**: Lógica de negócio e manipulação dos dados.
* **Modules/Views**: Exibição final ao usuário.

---

## 1. Definindo uma Model

Toda tabela deve ter uma classe na pasta **Models** que herda de `Database`.

```php
<?php
// Modules/Models/MinhaClasse.php

class MinhaClasse extends Database
{
    public function __construct()
    {
        $this->tabela = "minha_tabela";
        $this->chave_primaria = "minha_tabela_id";
        $this->order_by = "nome_da_coluna ASC";
        $this->join = [];
    }

    /**
     * Atalho estático para iniciar uma consulta fluida
     */
    public static function query() {
        return new self();
    }
}
?>
```

---

## 2. Operações de Leitura (Select)

O ORM utiliza **Interface Fluida**, permitindo encadear métodos para criar filtros.

### Listar vários registros
```php
// No seu Controller
$model = new MinhaClasse();

$resultados = $model->and('status', 'ativo')
                    ->limit(10)
                    ->list(); // ou ->listar()

foreach ($resultados as $item) {
    echo $item->nome_da_coluna;
}
```

### Buscar um único registro
```php
// Busca pelo ID da chave primária
$registro = MinhaClasse::query()->pk(1)->find(); // ou ->consultar()

if($registro->minha_tabela_id) {
    echo "Encontrado: " . $registro->nome_da_coluna;
}
```

---

## 3. Operações de Escrita (Insert/Update)

O método `save()` decide automaticamente se deve cadastrar ou alterar baseado na presença da Chave Primária.

### Inserir Novo
```php
$obj = new MinhaClasse();
$obj->coluna_nome = "Exemplo de Nome";
$obj->coluna_valor = 150.00;
$obj->save(); // Realiza INSERT automaticamente

echo "Gerado ID: " . $obj->last_id;
```

### Alterar Existente
```php
$obj = new MinhaClasse();
$obj->pk(10); // Carrega o ID 10 para alteração
$obj->coluna_nome = "Nome Atualizado";
$obj->save(); // Realiza UPDATE automaticamente
```

---

## 4. Filtros Avançados (Where)

| Método | Função | Exemplo |
| :--- | :--- | :--- |
| **and()** | Filtro AND | `->and('idade', 18, '>')` |
| **or()** | Filtro OR | `->or('tipo', 'admin')` |
| **limit()** | Limite de linhas | `->limit(5)` |
| **orderBy()** | Ordenação | `->orderBy('data DESC')` |
| **columns()** | Seleção de colunas | `->columns(['id', 'nome'])` |

---

## 5. Upload de Arquivos

A classe `UploadFile` simplifica a gestão de mídias, renomeando arquivos e criando pastas automaticamente.

### Upload de Arquivos (Restrito)
```php
try {
    $path = "uploads/documentos/";
    $file = $_FILES['arquivo_form'];
    $extensoes = "PDF, DOCX, JPG";

    // Retorna o nome do arquivo salvo ou lança exceção
    $nomeSalvo = UploadFile::sendRestrictedFiles($path, $file, "default", $extensoes);
    
    // Salva o nome no banco
    $obj = new MinhaClasse();
    $obj->documento_url = $nomeSalvo;
    $obj->save();

} catch (Exception $e) {
    echo "Erro no upload: " . $e->getMessage();
}
```

---

## 6. Depuração (Debug)

Se os dados não estiverem vindo como esperado, use o `debug()` para ver o SQL gerado em tempo real.

```php
MinhaClasse::query()
    ->and('nome', 'João', 'LIKE')
    ->debug() // Interrompe o script e exibe o SQL e os Parâmetros
    ->list();
```

---

## 7. Resumo de Comandos +## 7. Métricas de Performance + +Para analisar o tempo de execução de suas consultas, você pode habilitar o log de métricas.

```php 
MinhaClasse::query()
->logMetrics(true) // Habilita o log de tempo de execução
->and('status', 'ativo')
->list(); 
```

+Quando ativado, o ORM medirá o tempo que a consulta leva para ser executada no banco de dados e enviará essa informação para um coletor de métricas (\App\Shared\Helpers\MetricCollector). Isso é útil para identificar gargalos de performance. + +--- + +## 8. Resumo de Comandos

| Ação | Core (Inglês) | Wrapper (Português) |
| :--- | :--- | :--- |
| **Listar** | `list()` | `listar()` |
| **Buscar** | `find()` | `consultar()` |
| **Criar** | `create()` | `cadastrar()` |
| **Editar** | `update()` | `alterar()` |
| **Apagar** | `delete()` | `deletar()` |
| **Salvar** | `save()` | `salvar()` |
| **Metricas** | `logMetrics()` | `logMetrics()` |