# DIALOG.JS — DOCUMENTAÇÃO DE USO

Este módulo fornece um sistema de diálogos reutilizável, seguro (sem XSS) e desacoplado do HTML inline.
O diálogo é acionado diretamente em um elemento DOM através do método:

```javascript
element.dialog(options)
```

---

## ESTRUTURA HTML ESPERADA
O elemento deve conter a estrutura abaixo para que o dialog.js funcione corretamente:

```html
<div id="appDialog" class="dialog hidden">
    <div class="dialog-backdrop"></div>

    <div class="dialog-box">
        <div class="dialog-header">
            <h3 class="dialog-title"></h3>
            <button class="dialog-close">&times;</button>
        </div>

        <div class="dialog-body">
            <p class="dialog-message"></p>
            <div class="dialog-input-wrapper"></div>
        </div>

        <div class="dialog-footer">
            <button class="dialog-btn dialog-cancel">Cancelar</button>
            <button class="dialog-btn dialog-approve">Confirmar</button>
        </div>
    </div>
</div>
```

---

## FORMA BÁSICA DE USO

```javascript
const dialog = document.getElementById('appDialog')

dialog.dialog({
    title: 'Título do diálogo',
    message: 'Mensagem exibida ao usuário'
})
```

> Se nenhum **type** for informado, o diálogo será exibido com fundo branco e texto preto.

---

## TIPOS DE DIÁLOGO (VISUAL)
O parâmetro `type` altera apenas o estilo visual. Valores aceitos:
* `alert`
* `danger`
* `success`
* `info`

**Exemplo:**

```javascript
dialog.dialog({
    title: 'Atenção',
    message: 'Esta ação requer cuidado.',
    type: 'alert'
})
```

---

## DIÁLOGO DE CONFIRMAÇÃO (APROVAR / CANCELAR)

```javascript
dialog.dialog({
    title: 'Excluir cliente?',
    message: 'Esta ação não pode ser desfeita.',
    type: 'danger',
    onApprove: () => {
        console.log('Usuário confirmou')
    },
    onCancel: () => {
        console.log('Usuário cancelou')
    }
})
```

---

## DIÁLOGO COM INPUT (OPCIONAL)
O input só será criado se `input.show === true`.

```javascript
dialog.dialog({
    title: 'Ir para página',
    message: 'Digite um número entre 1 e 20',
    input: {
        show: true,
        type: 'number',
        min: 1,
        max: 20,
        placeholder: 'Ex: 5'
    },
    onApprove: (value) => {
        console.log('Valor informado:', value)
    }
})
```

---

## CALLBACKS DISPONÍVEIS
Todos os callbacks são opcionais.

```javascript
dialog.dialog({
    onOpen: () => {},
    onClose: () => {},
    onApprove: (value) => {},
    onCancel: () => {}
})
```

* **onOpen**: Executado quando o diálogo é exibido.
* **onClose**: Executado sempre que o diálogo é fechado.
* **onApprove(value)**: Executado ao clicar em "Confirmar". Recebe o valor do input ou null.
* **onCancel**: Executado ao clicar em "Cancelar", no backdrop ou ao pressionar ESC.

---

## FECHAMENTO AUTOMÁTICO
O diálogo é fechado automaticamente quando:
* O botão **Cancelar** é clicado
* O botão **Confirmar** é clicado
* O usuário clica no **backdrop**
* O usuário pressiona a tecla **ESC**

---

## EXEMPLO REAL — PAGINAÇÃO

```javascript
document.getElementById('btnPage').addEventListener('click', () => {
    dialog.dialog({
        title: 'Ir para página',
        message: 'Digite o número da página desejada',
        input: {
            show: true,
            type: 'number',
            min: 1,
            max: totalPages
        },
        onApprove: (page) => {
            if (!page) return
            goToPage(page)
        }
    })
})
```

---

## BOAS PRÁTICAS
* **✔** Nunca use onclick inline
* **✔** Nunca injete HTML via innerHTML
* **✔** Use sempre textContent (já aplicado internamente)
* **✔** Centralize o dialog no Layout
* **✔** Reutilize o mesmo elemento