import { getConfig } from '../shared/config.js'
import { initMasks, unmaskForm, validateForm } from '../shared/mask.js'

export function init() {

    // =====================================================
    // CONFIG
    // =====================================================
    const AppConfig = getConfig()
    const clienteApi = AppConfig.apiEndpoints.cliente
    const csrfToken = AppConfig.csrfToken

    // =====================================================
    // ELEMENTOS
    // =====================================================
    const form = document.getElementById('client-form')
    const toast = document.getElementById('toast')
    const submitBtn = document.getElementById('submit-btn')
    const submitText = document.getElementById('submit-text')
    const clientIdInput = document.getElementById('cliente_id')

    if (!form || !submitBtn || !clientIdInput) return
    initMasks(form)
    // =====================================================
    // TOAST
    // =====================================================
    function showToast(message, isError = false) {
        if (!toast) return

        toast.textContent = message
        toast.style.backgroundColor = isError ? '#ef4444' : '#22c55e'
        toast.style.display = 'block'
        toast.style.opacity = 1

        setTimeout(() => {
            toast.style.opacity = 0
            setTimeout(() => toast.style.display = 'none', 400)
        }, 4000)
    }



    // =====================================================
    // SUBMIT
    // =====================================================
    form.addEventListener('submit', async e => {
        e.preventDefault()

        const originalText = submitText.textContent
        submitBtn.disabled = true
        submitText.textContent = 'Salvando...'

        // valida antes
        if (!validateForm(form)) {
            submitBtn.disabled = false
            submitText.textContent = originalText
            return
        }

        // pega dados já "unmasked"
        let data = unmaskForm(form)

        const clientId = clientIdInput.value

        // CPF/CNPJ desabilitado na edição
        if (clientId) {
            const cpfInput = document.getElementById('cliente_cpf')
            if (cpfInput?.disabled) {
                data.cliente_cpf = cpfInput.value.replace(/\D/g, '')
            }
        }

        const url = clientId
            ? `${clienteApi.update}/${clientId}`
            : clienteApi.create

        try {
            const response = await fetch(url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': csrfToken
                },
                body: JSON.stringify(data)
            })

            const result = await response.json()

            if (response.ok && result.success) {
                showToast(result.message || 'Operação realizada com sucesso!')

                if (!clientId) {
                    setTimeout(() => {
                        window.location.href = `${AppConfig.baseUrl}/cliente`
                    }, 1500)
                }
            } else {
                showToast(result.message || 'Ocorreu um erro.', true)
            }

        } catch (err) {
            console.error(err)
            showToast('Erro de conexão.', true)
        } finally {
            submitBtn.disabled = false
            submitText.textContent = originalText
        }
    })

}
