import { getConfig } from '../shared/config.js'

export function init() {

    // =====================================================
    // CONFIG
    // =====================================================
    const AppConfig = getConfig()
    const contasBancariasApi = AppConfig.apiEndpoints.contasBancarias
    const csrfToken = AppConfig.csrfToken

    // =====================================================
    // ELEMENTOS
    // =====================================================
    const form = document.getElementById('contas-bancarias-form')
    const toast = document.getElementById('toast')
    const submitBtn = document.getElementById('submit-btn')
    const submitText = document.getElementById('submit-text')
    const contaBancariaIdInput = document.getElementById('conta-bancaria-id')

    if (!form || !submitBtn || !contaBancariaIdInput) return

    // =====================================================
    // TOAST
    // =====================================================
    function showToast(message, isError = false) {
        if (!toast) return

        toast.textContent = message
        toast.style.backgroundColor = isError ? '#ef4444' : '#22c55e'
        toast.style.display = 'block'
        toast.style.opacity = 1

        setTimeout(() => {
            toast.style.opacity = 0
            setTimeout(() => toast.style.display = 'none', 400)
        }, 4000)
    }

    // =====================================================
    // SUBMIT
    // =====================================================
    form.addEventListener('submit', async e => {
        e.preventDefault()

        const originalText = submitText.textContent
        submitBtn.disabled = true
        submitText.textContent = 'Salvando...'

        const formData = new FormData(form)
        const data = Object.fromEntries(formData.entries())
        const contaBancariaId = contaBancariaIdInput.value

        const url = contaBancariaId
            ? `${contasBancariasApi.update}/${contaBancariaId}`
            : contasBancariasApi.create

        try {
            const response = await fetch(url, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': csrfToken
                },
                body: JSON.stringify(data)
            })

            const result = await response.json()

            if (response.ok && result.success) {
                showToast(result.message || 'Operação realizada com sucesso!')

                if (!contaBancariaId) {
                    setTimeout(() => {
                        window.location.href = `${AppConfig.baseUrl}/contasbancarias`
                    }, 1500)
                }
            } else {
                showToast(result.message || 'Ocorreu um erro.', true)
            }

        } catch (err) {
            console.error(err)
            showToast('Erro de conexão.', true)
        } finally {
            submitBtn.disabled = false
            submitText.textContent = originalText
        }
    })
}
