import { getConfig } from '../../shared/config.js'

export function init() {

    const form = document.getElementById('categoria-form')
    const toast = document.getElementById('toast')

    // =====================================================
    // CONFIG
    // =====================================================
    let AppConfig
    try {
        AppConfig = getConfig()
    } catch {
        return
    }

    const endpoints = AppConfig.apiEndpoints.categoriaProduto
    const csrfToken = AppConfig.csrfToken

    if (!form) return

    // =====================================================
    // HELPERS
    // =====================================================
    function showToast(message, isError = false) {
        if (!toast) return
        toast.textContent = message
        toast.style.backgroundColor = isError ? '#ef4444' : '#22c55e'
        toast.style.display = 'block'
        toast.style.opacity = 1
        setTimeout(() => {
            toast.style.opacity = 0
            setTimeout(() => (toast.style.display = 'none'), 400)
        }, 4000)
    }

    // Submit Logic
    form.addEventListener('submit', async (e) => {
        e.preventDefault()

        const submitBtn = document.getElementById('submit-btn')
        const originalBtnText = submitBtn.innerHTML
        submitBtn.disabled = true
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Salvando...'

        const formData = new FormData(form)
        const data = Object.fromEntries(formData.entries())

        const id = data.produto_categoria_id
        const isUpdate = !!id
        const endpoint = isUpdate 
            ? `${endpoints.update}/${id}`
            : endpoints.create

        try {
            const response = await fetch(endpoint, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': csrfToken
                },
                body: JSON.stringify(data)
            })

            const result = await response.json()

            if (result.success) {
                showToast(result.message)
                if (!isUpdate) {
                    form.reset() // Limpa form apenas se for criação
                }
                
                // Redirecionar para a lista
                // Usamos AppConfig.baseUrl se disponível ou construimos relativo
                setTimeout(() => {
                     // Assume-se que o endpoint 'list' é algo como BASE_URL/estoque/categoriaproduto/list
                     // Para voltar para a tela de listagem (HTML), removemos o sufixo '/list' ou usamos hardcoded
                     // Se o padrão for BASE_URL + /estoque/categoriaproduto, podemos derivar:
                     const listUrl = endpoints.list.replace('/list', '')
                     window.location.href = listUrl;
                }, 1000);
                
            } else {
                showToast(result.message || 'Erro ao salvar.', true)
            }
        } catch (err) {
            console.error(err)
            showToast('Erro de conexão.', true)
        } finally {
            submitBtn.disabled = false
            submitBtn.innerHTML = originalBtnText
        }
    })
}
