import { getConfig } from '../../shared/config.js'

export function init() {

    const form = document.getElementById('produto-form')
    const toast = document.getElementById('toast')

    let AppConfig
    try {
        AppConfig = getConfig()
    } catch {
        return
    }

    const endpoints = AppConfig.apiEndpoints.produtos
    const categoriaEndpoints = AppConfig.apiEndpoints.categoriaProduto // Fetching Categoria Endpoints
    const unidadeEndpoints = AppConfig.apiEndpoints.unidadeMedida // Fetching Unidade Endpoints
    const csrfToken = AppConfig.csrfToken

    if (!form) return

    function showToast(message, isError = false) {
        if (!toast) return
        toast.textContent = message
        toast.style.backgroundColor = isError ? '#ef4444' : '#22c55e'
        toast.style.display = 'block'
        toast.style.opacity = 1
        setTimeout(() => {
            toast.style.opacity = 0
            setTimeout(() => (toast.style.display = 'none'), 400)
        }, 4000)
    }

    // --- Searchable Select Logic ---

    async function loadOptions(endpoint, searchInputId, hiddenInputId, dropdownId, nameField, idField, initialId) {
        const searchInput = document.getElementById(searchInputId)
        const hiddenInput = document.getElementById(hiddenInputId)
        const dropdown = document.getElementById(dropdownId)
        let options = []

        if(!searchInput || !dropdown) return;

        function renderOptions(filtered, showAll = false) {
            dropdown.innerHTML = ''
            
            // Should we limit to 5 if showing all/default?
            // User asked: "pode puxar todos mas quando pesquisar ele filtra", implies show all or many on focus
            // But also said "mostrar os 5 primeiros" -> possibly meaning "show at least 5"
            // Let's safe bet: show all if filtered is small, or limit to 100 if huge.
            // But specifically asked "mostrar os 5 primeiros" -> maybe means "initial view"
            
            const limit = showAll ? filtered.length : filtered.length 
            const listToRender = filtered.slice(0, 100) // Render up to 100 to stay snappy

            if(listToRender.length === 0) {
                const div = document.createElement('div')
                div.className = 'px-4 py-2 text-sm text-gray-500'
                div.textContent = 'Nenhum resultado.'
                dropdown.appendChild(div)
                return
            }
            
            listToRender.forEach(item => {
                const div = document.createElement('div')
                div.className = 'px-4 py-2 hover:bg-blue-50 cursor-pointer text-sm text-gray-700 border-b border-gray-50 last:border-0'
                div.textContent = item[nameField]
                div.addEventListener('click', () => {
                    searchInput.value = item[nameField]
                    hiddenInput.value = item[idField]
                    dropdown.classList.add('hidden')
                })
                dropdown.appendChild(div)
            })
        }

        try {
            const response = await fetch(`${endpoint.list}?page=1&limit=1000`)
            const result = await response.json()
            if(result.success && result.pagination && result.pagination.data) {
                options = result.pagination.data
                
                // Initial Render (Top 5 as requested, or just render default list ready)
                // We won't show it yet (hidden class is there), but content is ready.
                // Or user meant "show 5 options in the input"? No, meant in dropdown.
                renderOptions(options) 
            }
        } catch(e) {
            console.error("Failed to load options", e)
        }

        if(initialId && options.length > 0) {
            const initialItem = options.find(i => String(i[idField]) === String(initialId))
            if(initialItem) {
                searchInput.value = initialItem[nameField]
                hiddenInput.value = initialItem[idField]
            }
        }

        searchInput.addEventListener('input', (e) => {
            const term = e.target.value.toLowerCase()
            const filtered = options.filter(item => String(item[nameField]).toLowerCase().includes(term))
            renderOptions(filtered)
            dropdown.classList.remove('hidden')
            
            if(term === '') hiddenInput.value = ''
        })

        searchInput.addEventListener('focus', () => {
            // Refilter in case value changed or wasn't empty
            const term = searchInput.value.toLowerCase()
            const filtered = options.filter(item => String(item[nameField]).toLowerCase().includes(term))
            
            // If empty, show all (or top 5 if explicitly requested, but usually all is better for selection)
            // "deveria pelo menos mostrar os 5 primeiros" -> I will ensure I render plenty.
            if (term === '') {
                 renderOptions(options)
            } else {
                 renderOptions(filtered)
            }
            dropdown.classList.remove('hidden')
        })

        document.addEventListener('click', (e) => {
            if(!e.target.closest(`#${searchInputId}`) && !e.target.closest(`#${dropdownId}`)) {
                dropdown.classList.add('hidden')
            }
        })
    }

    // Initialize Searchable Selects
    const currentCategoria = document.getElementById('current_categoria_id')?.value
    const currentUnidade = document.getElementById('current_unidade_id')?.value

    if(categoriaEndpoints) {
        loadOptions(
            categoriaEndpoints, 
            'produto_categoria_search', 
            'produto_categoria', 
            'categoria-dropdown', 
            'produto_categoria_nome', 
            'produto_categoria_id',
            currentCategoria
        )
    }

    if(unidadeEndpoints) {
        loadOptions(
            unidadeEndpoints, 
            'produto_unidade_search', 
            'produto_unidade_medida', 
            'unidade-dropdown', 
            'unidade_medida_nome', 
            'unidade_medida_id',
            currentUnidade
        )
    }


    // --- Pricing Calculation Logic ---

    const inputCost = document.getElementById('produto_valor')
    const inputMargin = document.getElementById('produto_margemLucro')
    const inputSale = document.getElementById('produto_valorVenda')

    function calculate() {
        const cost = parseFloat(inputCost.value) || 0
        const margin = parseFloat(inputMargin.value) || 0
        const sale = parseFloat(inputSale.value) || 0
        
        // Identify which input triggered the event is hard if generic, 
        // using specific handlers is better.
    }

    function onCostChange() {
        const cost = parseFloat(inputCost.value)
        const margin = parseFloat(inputMargin.value)
        const sale = parseFloat(inputSale.value)

        if (!isNaN(cost) && !isNaN(margin)) {
            // Cost + Margin -> Sale
            const newSale = cost * (1 + (margin / 100))
            inputSale.value = newSale.toFixed(2)
        } else if (!isNaN(cost) && !isNaN(sale) && isNaN(margin)) {
            // Cost + Sale -> Margin (if Margin is empty)
             const newMargin = ((sale - cost) / cost) * 100
             inputMargin.value = newMargin.toFixed(2)
        }
    }

    function onMarginChange() {
        const cost = parseFloat(inputCost.value)
        const margin = parseFloat(inputMargin.value)
        const sale = parseFloat(inputSale.value)

        if (!isNaN(cost) && !isNaN(margin)) {
             // Margin + Cost -> Sale
             const newSale = cost * (1 + (margin / 100))
             inputSale.value = newSale.toFixed(2)
        } else if (!isNaN(margin) && !isNaN(sale) && isNaN(cost)) {
             // Margin + Sale -> Cost
             const newCost = sale / (1 + (margin / 100))
             inputCost.value = newCost.toFixed(2)
        }
    }
    
    function onSaleChange() {
        const cost = parseFloat(inputCost.value)
        const margin = parseFloat(inputMargin.value)
        const sale = parseFloat(inputSale.value)

        if (!isNaN(cost) && !isNaN(sale)) {
            // Sale + Cost -> Margin
            if(cost !== 0) {
                const newMargin = ((sale - cost) / cost) * 100
                inputMargin.value = newMargin.toFixed(2)
            }
        } else if (!isNaN(sale) && !isNaN(margin) && isNaN(cost)) {
             // Sale + Margin -> Cost
             const newCost = sale / (1 + (margin / 100))
             inputCost.value = newCost.toFixed(2)
        }
    }

    if(inputCost && inputMargin && inputSale) {
        inputCost.addEventListener('input', () => {
             // Logic: If Margin is present, update Sale. If Sale is present (and margin empty? or just standard flow?), update Margin?
             // Standard flow usually: Cost + Margin = Sale.
             // If I define Cost, and I have Margin, Sale updates.
             // If I define Cost, and I have Sale (but no Margin logic?), Margin updates.
             
             // Simplest heuristic based on "2 fields filled":
             const margin = parseFloat(inputMargin.value)
             const sale = parseFloat(inputSale.value)

             if (!isNaN(margin)) {
                 inputSale.value = (parseFloat(inputCost.value) * (1 + margin/100)).toFixed(2)
             } else if (!isNaN(sale)) {
                 const c = parseFloat(inputCost.value)
                 if(c !== 0) inputMargin.value = (((sale - c) / c) * 100).toFixed(2)
             }
        })

        inputMargin.addEventListener('input', () => {
             const cost = parseFloat(inputCost.value)
             const sale = parseFloat(inputSale.value)
             
             if(!isNaN(cost)) {
                 inputSale.value = (cost * (1 + parseFloat(inputMargin.value)/100)).toFixed(2)
             } else if (!isNaN(sale)) {
                 inputCost.value = (sale / (1 + parseFloat(inputMargin.value)/100)).toFixed(2)
             }
        })

        inputSale.addEventListener('input', () => {
             const cost = parseFloat(inputCost.value)
             const margin = parseFloat(inputMargin.value)
             
             if(!isNaN(cost)) {
                 if(cost !== 0) inputMargin.value = (((parseFloat(inputSale.value) - cost) / cost) * 100).toFixed(2)
             } else if (!isNaN(margin)) {
                 inputCost.value = (parseFloat(inputSale.value) / (1 + margin/100)).toFixed(2)
             }
        })
    }

    // --- Form Submission ---

    form.addEventListener('submit', async (e) => {
        e.preventDefault()

        const submitBtn = document.getElementById('submit-btn')
        const originalBtnText = submitBtn.innerHTML
        submitBtn.disabled = true
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Salvando...'

        const formData = new FormData(form)
        const data = Object.fromEntries(formData.entries())

        // Debug: Log what we're sending
        console.log('Form data being sent:', data)
        console.log('Categoria ID:', data.produto_categoria)
        console.log('Unidade ID:', data.produto_unidade_medida)

        // Validate required foreign keys
        if (!data.produto_categoria || data.produto_categoria === '') {
            showToast('Por favor, selecione uma categoria.', true)
            submitBtn.disabled = false
            submitBtn.innerHTML = originalBtnText
            return
        }

        if (!data.produto_unidade_medida || data.produto_unidade_medida === '') {
            showToast('Por favor, selecione uma unidade de medida.', true)
            submitBtn.disabled = false
            submitBtn.innerHTML = originalBtnText
            return
        }

        const id = data.produto_id
        const isUpdate = !!id
        const endpoint = isUpdate 
            ? `${endpoints.update}/${id}`
            : endpoints.create

        try {
            const response = await fetch(endpoint, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': csrfToken
                },
                body: JSON.stringify(data)
            })

            const result = await response.json()

            if (result.success) {
                showToast(result.message)
                if (!isUpdate) {
                    form.reset()
                    // Clear search inputs too
                    document.getElementById('produto_categoria_search').value = ''
                    document.getElementById('produto_unidade_search').value = ''
                }
                
                setTimeout(() => {
                     const listUrl = endpoints.list.replace('/list', '')
                     window.location.href = listUrl;
                }, 1000);
                
            } else {
                showToast(result.message || 'Erro ao salvar.', true)
            }
        } catch (err) {
            console.error(err)
            showToast('Erro de conexão.', true)
        } finally {
            submitBtn.disabled = false
            submitBtn.innerHTML = originalBtnText
        }
    })
}
