import { getConfig } from '../../shared/config.js'

export function init() {

    let AppConfig
    try {
        AppConfig = getConfig()
    } catch {
        return
    }

    const endpoints = AppConfig.apiEndpoints.unidadeMedida
    const csrfToken = AppConfig.csrfToken

    const listTable = document.getElementById('unidade-list')
    const paginationControls = document.getElementById('pagination-controls')
    const toast = document.getElementById('toast')
    const dialogEl = document.getElementById('appDialog')

    if (!listTable || !paginationControls) return
    if (!dialogEl || typeof dialogEl.dialog !== 'function') return

    function showToast(message, isError = false) {
        if (!toast) return
        toast.textContent = message
        toast.style.backgroundColor = isError ? '#ef4444' : '#22c55e'
        toast.style.display = 'block'
        toast.style.opacity = 1
        setTimeout(() => {
            toast.style.opacity = 0
            setTimeout(() => (toast.style.display = 'none'), 400)
        }, 4000)
    }

    let currentController = null

    async function fetchItems(page = 1) {
        if (currentController) currentController.abort()
        currentController = new AbortController()

        try {
            const response = await fetch(
                `${endpoints.list}?page=${page}`,
                { signal: currentController.signal }
            )
            const result = await response.json()

            if (currentController.signal.aborted) return

            if (result.success && result.pagination) {
                renderItems(result.pagination.data)
                renderPagination(result.pagination)
            } else {
                showToast(result.message || 'Erro ao carregar unidades.', true)
            }
        } catch (err) {
            if (err.name !== 'AbortError') {
                showToast('Erro de conexão.', true)
            }
        }
    }

    function renderItems(items) {
        listTable.innerHTML = ''

        if (!items || items.length === 0) {
            listTable.innerHTML = `
                <tr>
                    <td colspan="3" class="text-center py-4">
                        Nenhuma unidade cadastrada.
                    </td>
                </tr>
            `
            return
        }

        items.forEach(item => {
            const tr = document.createElement('tr')
            tr.dataset.id = item.unidade_medida_id

            tr.innerHTML = `
                <td class="px-6 py-4 text-sm font-medium">${item.unidade_medida_id}</td>
                <td class="px-6 py-4 text-sm">${escapeHtml(item.unidade_medida_nome)}</td>
                <td class="px-6 py-4 text-right text-sm">
                    <a href="${endpoints.form}/${item.unidade_medida_id}"
                       class="text-indigo-600 hover:text-indigo-900">
                        Editar
                    </a>
                    <button
                        type="button"
                        class="ml-4 text-red-600 hover:text-red-900 delete-btn">
                        Excluir
                    </button>
                </td>
            `
            listTable.appendChild(tr)
        })
    }

    function createPageButton(page, current) {
        const btn = document.createElement('button')
        btn.textContent = page
        btn.dataset.page = page
        btn.className = page === current
            ? 'px-4 py-2 bg-blue-600 text-white rounded-md'
            : 'px-4 py-2 bg-gray-200 rounded-md hover:bg-gray-300'
        btn.disabled = page === current
        return btn
    }

    function renderPagination(pagination) {
        paginationControls.innerHTML = ''
        if (!pagination || pagination.total_pages <= 1) return

        const { current_page, total_pages } = pagination

        const prev = document.createElement('button')
        prev.innerHTML = '&laquo; Anterior'
        prev.dataset.page = current_page - 1
        prev.disabled = current_page <= 1
        prev.className = 'px-4 py-2 bg-gray-200 rounded-md disabled:opacity-50'
        paginationControls.appendChild(prev)

        for (let i = 1; i <= total_pages; i++) {
            paginationControls.appendChild(createPageButton(i, current_page))
        }

        const next = document.createElement('button')
        next.innerHTML = 'Próximo &raquo;'
        next.dataset.page = current_page + 1
        next.disabled = current_page >= total_pages
        next.className = 'px-4 py-2 bg-gray-200 rounded-md disabled:opacity-50'
        paginationControls.appendChild(next)
    }

    paginationControls.addEventListener('click', e => {
        if (e.target.dataset.page) {
            fetchItems(parseInt(e.target.dataset.page, 10))
        }
    })

    listTable.addEventListener('click', e => {
        const btn = e.target.closest('.delete-btn')
        if (!btn) return
        const row = btn.closest('tr')
        const id = row?.dataset.id
        if (!id) return

        dialogEl.dialog({
            title: 'Excluir unidade?',
            message: 'Esta ação não pode ser desfeita.',
            type: 'danger',
            onApprove: async () => {
                try {
                    const response = await fetch(
                        `${endpoints.delete}/${id}`,
                        {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'X-CSRF-TOKEN': csrfToken
                            }
                        }
                    )
                    const result = await response.json()
                    if (response.ok && result.success) {
                        showToast(result.message)
                        fetchItems()
                    } else {
                        showToast(result.message || 'Erro ao excluir.', true)
                    }
                } catch {
                    showToast('Erro de conexão.', true)
                }
            }
        })
    })

    function escapeHtml(str) {
        if (str == null) return ''
        const p = document.createElement('p')
        p.textContent = str
        return p.innerHTML
    }

    fetchItems(1)
}
