(function () {

    /**
     * Inicializa o dialog em um elemento
     * @param {Object} options
     */
    HTMLElement.prototype.dialog = function (options = {}) {

        const dialog = this

        if (!dialog.classList.contains('dialog')) {
            console.warn('Elemento não é um dialog válido')
            return
        }

        // ===============================
        // ELEMENTOS
        // ===============================
        const backdrop = dialog.querySelector('.dialog-backdrop')
        const titleEl = dialog.querySelector('.dialog-title')
        const messageEl = dialog.querySelector('.dialog-message')
        const inputWrapper = dialog.querySelector('.dialog-input-wrapper')
        const btnApprove = dialog.querySelector('.dialog-approve')
        const btnCancel = dialog.querySelector('.dialog-cancel')
        const btnClose = dialog.querySelector('.dialog-close')

        // ===============================
        // DEFAULTS
        // ===============================
        const {
            title = '',
            message = '',
            type = '',
            input = { show: false },
            onOpen = null,
            onClose = null,
            onApprove = null,
            onCancel = null
        } = options

        // ===============================
        // RESET
        // ===============================
        dialog.classList.remove('alert', 'danger', 'success', 'info')
        inputWrapper.innerHTML = ''

        // ===============================
        // CONTENT
        // ===============================
        titleEl.textContent = title
        messageEl.textContent = message

        if (type) {
            dialog.classList.add(type)
        }

        // ===============================
        // INPUT (OPCIONAL)
        // ===============================
        let inputEl = null

        if (input && input.show === true) {
            inputEl = document.createElement('input')
            inputEl.className = 'app-dialog-input'
            inputEl.type = input.type || 'text'

            if (input.placeholder) inputEl.placeholder = input.placeholder
            if (input.min !== undefined) inputEl.min = input.min
            if (input.max !== undefined) inputEl.max = input.max

            inputWrapper.appendChild(inputEl)
        }

        // ===============================
        // OPEN
        // ===============================
        dialog.classList.remove('hidden')

        if (typeof onOpen === 'function') {
            onOpen()
        }

        // ===============================
        // CLOSE HANDLER
        // ===============================
        const close = (callback) => {
            dialog.classList.add('hidden')

            dialog.removeEventListener('keydown', escHandler)
            backdrop.removeEventListener('click', backdropHandler)

            if (typeof callback === 'function') callback()
            if (typeof onClose === 'function') onClose()
        }

        // ===============================
        // EVENTS
        // ===============================
        const approveHandler = () => {
            const value = inputEl ? inputEl.value : null

            if (typeof onApprove === 'function') {
                onApprove(value)
            }

            close()
        }

        const cancelHandler = () => {
            if (typeof onCancel === 'function') {
                onCancel()
            }

            close()
        }

        const backdropHandler = (e) => {
            if (e.target === backdrop) {
                cancelHandler()
            }
        }

        const escHandler = (e) => {
            if (e.key === 'Escape') {
                cancelHandler()
            }
        }

        // ===============================
        // BIND
        // ===============================
        btnApprove.onclick = approveHandler
        btnCancel.onclick = cancelHandler
        if (btnClose) btnClose.onclick = cancelHandler

        backdrop.addEventListener('click', backdropHandler)
        document.addEventListener('keydown', escHandler)

        if (inputEl) {
            inputEl.focus()
        }
    }

})()
