// assets/js/shared/mask.js

const masks = {
    cep(value) {
        return value.replace(/\D/g, '')
            .slice(0, 8)
            .replace(/(\d{5})(\d)/, '$1-$2')
    },

    cpf(value) {
        return value.replace(/\D/g, '')
            .slice(0, 11)
            .replace(/(\d{3})(\d)/, '$1.$2')
            .replace(/(\d{3})(\d)/, '$1.$2')
            .replace(/(\d{3})(\d{1,2})$/, '$1-$2')
    },

    cnpj(value) {
        return value.replace(/\D/g, '')
            .slice(0, 14)
            .replace(/(\d{2})(\d)/, '$1.$2')
            .replace(/(\d{3})(\d)/, '$1.$2')
            .replace(/(\d{3})(\d)/, '$1/$2')
            .replace(/(\d{4})(\d)/, '$1-$2')
    },

    cpf_cnpj(value) {
        const clean = value.replace(/\D/g, '')
        return clean.length <= 11
            ? masks.cpf(clean)
            : masks.cnpj(clean)
    },

    phone(value) {
        const v = value.replace(/\D/g, '').slice(0, 11)
        return v.length > 10
            ? v.replace(/(\d{2})(\d{5})(\d{4})/, '($1) $2-$3')
            : v.replace(/(\d{2})(\d{4})(\d{4})/, '($1) $2-$3')
    },

    money(value) {
        let v = value.replace(/\D/g, '')
        v = (v / 100).toFixed(2).replace('.', ',')
        return 'R$ ' + v.replace(/(\d)(?=(\d{3})+(?!\d))/g, '$1.')
    }
}

// ================================
// VALIDADORES
// ================================
const validators = {
    cpf(value) {
        value = value.replace(/\D/g, '')
        if (value.length !== 11 || /^(\d)\1+$/.test(value)) return false

        let sum = 0
        for (let i = 0; i < 9; i++) sum += value[i] * (10 - i)
        let check = (sum * 10) % 11
        if (check === 10) check = 0
        if (check != value[9]) return false

        sum = 0
        for (let i = 0; i < 10; i++) sum += value[i] * (11 - i)
        check = (sum * 10) % 11
        if (check === 10) check = 0
        return check == value[10]
    },

    cnpj(value) {
        value = value.replace(/\D/g, '')
        if (value.length !== 14 || /^(\d)\1+$/.test(value)) return false

        const calc = (len) => {
            let sum = 0, pos = len - 7
            for (let i = len; i >= 1; i--) {
                sum += value[len - i] * pos--
                if (pos < 2) pos = 9
            }
            return sum % 11 < 2 ? 0 : 11 - (sum % 11)
        }

        return calc(12) == value[12] && calc(13) == value[13]
    },

    cpf_cnpj(value) {
        const v = value.replace(/\D/g, '')
        return v.length <= 11
            ? validators.cpf(v)
            : validators.cnpj(v)
    }
}

function unmaskMoney(value) {
    if (!value) return '0.00'

    // remove tudo exceto números, vírgula e ponto
    value = value.replace(/[^\d.,]/g, '')

    // se tem vírgula, assume padrão BR
    if (value.includes(',')) {
        value = value.replace(/\./g, '').replace(',', '.')
    }

    const num = parseFloat(value)

    if (isNaN(num)) return '0.00'

    // garante 2 casas decimais
    return num.toFixed(2)
}


// ================================
// PLUGIN
// ================================
export function initMasks(root = document) {
    root.querySelectorAll('[data-mask]').forEach(el => {
        const mask = el.dataset.mask
        el.addEventListener('input', () => {
            el.value = masks[mask]?.(el.value) ?? el.value
            clearError(el)
        })
        if (el.value) el.value = masks[mask]?.(el.value)
    })
}

export function unmaskForm(form) {
    const data = {}

    form.querySelectorAll('input, select, textarea').forEach(el => {
        if (!el.name) return

        const mask = el.dataset.unmask

        if (mask === 'money') {
            data[el.name] = unmaskMoney(el.value)
            return
        }

        if (mask === 'number') {
            data[el.name] = el.value.replace(/\D/g, '')
            return
        }

        // default: texto puro
        data[el.name] = el.value
    })

    return data
}


export function validateForm(form) {
    let valid = true

    form.querySelectorAll('[data-validate]').forEach(el => {
        const rule = el.dataset.validate
        if (!validators[rule]) return

        if (!validators[rule](el.value)) {
            showError(el, 'Valor inválido')
            valid = false
        }
    })

    return valid
}

// ================================
// UI Tailwind
// ================================
function showError(el, msg) {
    el.classList.add('border-red-500', 'ring-1', 'ring-red-500')
    el.setCustomValidity(msg)
}

function clearError(el) {
    el.classList.remove('border-red-500', 'ring-red-500')
    el.setCustomValidity('')
}
